<?php
set_time_limit(600);
include('connect.php');


// helper: fetch FPL history for one manager (returns associative array or null)


// Updated fetchFplHistory function
function fetchFplHistory($managerId)
{
    $url = "https://fantasy.premierleague.com/api/entry/{$managerId}/history/";

    // Use the reliable cURL function instead of file_get_contents
    $json = curl_fetch_url($url);

    if ($json === false) return null;

    $data = json_decode($json, true);
    return is_array($data) ? $data : null;
}

function updateManagerHistory($manager_id, $gw)
{
    global $db;
    // 1. Get picks data for that manager and gameweek
    $picks_url = "https://fantasy.premierleague.com/api/entry/$manager_id/event/$gw/picks/";
    $picks_data = curl_fetch_url($picks_url);

    if (!$picks_data) {
        echo "Error fetching picks for manager $manager_id (GW $gw)<br>";
        return;
    }

    $picks_data = json_decode($picks_data, true);
    if (!$picks_data || !isset($picks_data['entry_history'])) {
        echo "Invalid data for manager $manager_id (GW $gw)<br>";
        return;
    }

    // 2. Get live data for all players in that gameweek
    $live_url = "https://fantasy.premierleague.com/api/event/$gw/live/";
    $live_data = curl_fetch_url($live_url);
    if (!$live_data) {
        echo "Error fetching live data for GW $gw<br>";
        return;
    }

    $live_data = json_decode($live_data, true);
    $players_stats = [];
    foreach ($live_data['elements'] as $player) {
        $players_stats[$player['id']] = $player['stats'];
    }

    // 3. Get key data
    $points = $picks_data['entry_history']['points'];
    $transfer_cost = $picks_data['entry_history']['event_transfers_cost'];
    $net_points = $points - $transfer_cost;
    $bench_points = $picks_data['entry_history']['points_on_bench'];

    // 4. Get vice captain
    $vice_captain_id = null;
    foreach ($picks_data['picks'] as $pick) {
        if ($pick['is_vice_captain']) {
            $vice_captain_id = $pick['element'];
            break;
        }
    }

    $vice_captain_points = 0;
    if ($vice_captain_id && isset($players_stats[$vice_captain_id])) {
        $vice_captain_points = $players_stats[$vice_captain_id]['total_points'];
    }

    // 5. Calculate total goals scored by starting XI
    $team_goals = 0;
    foreach ($picks_data['picks'] as $pick) {
        if ($pick['position'] <= 11) {
            $player_id = $pick['element'];
            if (isset($players_stats[$player_id])) {
                $team_goals += $players_stats[$player_id]['goals_scored'];
            }
        }
    }

    // 6. Save to DB
    $stmt = $db->query("INSERT INTO manager_history (manager_id, event, points, transfer_cost, net_points, vice_captain_points, bench_points, team_goals) VALUES ('$manager_id', '$gw', '$points', '$transfer_cost', '$net_points', '$vice_captain_points', '$bench_points', '$team_goals') ON DUPLICATE KEY UPDATE points = VALUES(points), net_points = VALUES(net_points), vice_captain_points = VALUES(vice_captain_points), bench_points = VALUES(bench_points), team_goals = VALUES(team_goals)");

    echo "Updated manager $manager_id for GW $gw<br>";
}




// function updateManagerHistories()
// {
//     global $db;

//     // Get all unique manager IDs from teams
//     $managers = [];
//     $res = $db->query("SELECT manager1, manager2, manager3, manager4, manager5 FROM teams");
//     while ($row = mysqli_fetch_assoc($res)) {
//         foreach ($row as $m) {
//             if ($m && intval($m) > 0) {
//                 $managers[intval($m)] = true;
//             }
//         }
//     }

//     foreach (array_keys($managers) as $managerId) {
//         $history = fetchFplHistory($managerId);

//         if (!$history || !isset($history['current'])) continue;

//         foreach ($history['current'] as $week) {
//             $gw = intval($week['event']);
//             $points = intval($week['points'] ?? 0);
//             $transferCost = intval($week['event_transfers_cost'] ?? 0);
//             $net = $points - $transferCost;

//             // Insert or update row
//             $sp = $db->query("SELECT * FROM manager_history WHERE manager_id = '$managerId' AND event = '$gw'");
//             if (mysqli_num_rows($sp) >= 1) {
//                 $db->query("UPDATE manager_history SET points = '$points', transfer_cost = '$transferCost', net_points = '$net' WHERE manager_id = '$managerId' AND event = '$gw'");
//                 echo 'yes';
//             } else {
//                 $db->query("INSERT INTO manager_history (manager_id, event, points, transfer_cost, net_points)VALUES ('$managerId', '$gw', '$points', '$transferCost', '$net')");
//                 echo 'yes also';
//             }
//         }
//     }
//    updateAllResults();
// }

function multiFetchFplHistories($managerIds)
{
    $multi = curl_multi_init();
    $channels = [];
    $responses = [];

    // Create cURL handles for all managers
    foreach ($managerIds as $id) {
        $url = "https://fantasy.premierleague.com/api/entry/{$id}/history/";
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false
        ]);
        curl_multi_add_handle($multi, $ch);
        $channels[$id] = $ch;
    }

    // Run all requests in parallel
    $running = null;
    do {
        curl_multi_exec($multi, $running);
        curl_multi_select($multi);
    } while ($running > 0);

    // Collect all results
    foreach ($channels as $id => $ch) {
        $content = curl_multi_getcontent($ch);
        $responses[$id] = json_decode($content, true);
        curl_multi_remove_handle($multi, $ch);
        curl_close($ch);
    }

    curl_multi_close($multi);
    return $responses;
}


function fetchFplTeamData($managerId, $gw)
{
    $url = "https://fantasy.premierleague.com/api/entry/{$managerId}/event/{$gw}/picks/";
    $data = curl_fetch_url($url);
    return $data ? json_decode($data, true) : null;
}

function getViceCaptainPointsFast($teamData, $liveData)
{
    if (!$teamData || !isset($teamData['picks'])) return 0;
    if (!isset($liveData['elements'])) return 0;

    $viceCaptain = null;
    foreach ($teamData['picks'] as $pick) {
        if ($pick['is_vice_captain']) {
            $viceCaptain = $pick['element'];
            break;
        }
    }

    if (!$viceCaptain) return 0;

    foreach ($liveData['elements'] as $player) {
        if ($player['id'] == $viceCaptain) {
            return intval($player['stats']['total_points'] ?? 0);
        }
    }

    return 0;
}

function getTeamGoalsFast($teamData, $liveData)
{
    if (!$teamData || !isset($teamData['picks'])) return 0;
    if (!isset($liveData['elements'])) return 0;

    $goals = 0;
    foreach ($teamData['picks'] as $pick) {
        $pid = $pick['element'];
        foreach ($liveData['elements'] as $player) {
            if ($player['id'] == $pid) {
                $goals += intval($player['stats']['goals_scored'] ?? 0);
                break;
            }
        }
    }
    return $goals;
}

function multiCurlFetch($urls)
{
    $multiHandle = curl_multi_init();
    $curlHandles = [];
    $responses = [];

    // Init handles
    foreach ($urls as $key => $url) {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false
        ]);
        curl_multi_add_handle($multiHandle, $ch);
        $curlHandles[$key] = $ch;
    }

    // Run all requests
    $running = null;
    do {
        curl_multi_exec($multiHandle, $running);
        curl_multi_select($multiHandle);
    } while ($running > 0);

    // Collect responses
    foreach ($curlHandles as $key => $ch) {
        $responses[$key] = curl_multi_getcontent($ch);
        curl_multi_remove_handle($multiHandle, $ch);
        curl_close($ch);
    }

    curl_multi_close($multiHandle);
    return $responses;
}




function updateManagerHistories()
{
    global $db;

    // Step 1: Collect all manager IDs
    $managerIds = [];
    $res = $db->query("SELECT manager1, manager2, manager3, manager4, manager5 FROM teams");
    while ($row = mysqli_fetch_assoc($res)) {
        foreach ($row as $m) {
            $id = intval($m);
            if ($id > 0) $managerIds[$id] = true;
        }
    }
    $managerIds = array_keys($managerIds);

    // Step 2: Cache existing manager-history records
    $existing = [];
    $result = $db->query("SELECT manager_id, event FROM manager_history");
    while ($r = mysqli_fetch_assoc($result)) {
        $existing[$r['manager_id']][$r['event']] = true;
    }

    // Step 3: Fetch all manager histories in parallel
    $histories = multiFetchFplHistories($managerIds);

    // Step 4: Loop through each gameweek found in the histories
    $allGameweeks = [];
    foreach ($histories as $h) {
        if (!isset($h['current'])) continue;
        foreach ($h['current'] as $week) {
            $allGameweeks[intval($week['event'])] = true;
        }
    }

    $allGameweeks = array_keys($allGameweeks);
    sort($allGameweeks);

    foreach ($allGameweeks as $gw) {
        // Step 5: Fetch live data for this gameweek once
        $liveData = curl_fetch_url("https://fantasy.premierleague.com/api/event/{$gw}/live/");
        $liveData = $liveData ? json_decode($liveData, true) : [];

        // Step 6: Build all team URLs for parallel picks fetching
        $urls = [];
        foreach ($managerIds as $id) {
            $urls[$id] = "https://fantasy.premierleague.com/api/entry/{$id}/event/{$gw}/picks/";
        }

        $picksData = multiCurlFetch($urls); // parallel fetch for all managers

        // Step 7: Loop through all managers
        foreach ($managerIds as $managerId) {
            $history = $histories[$managerId]['current'] ?? [];
            $weekData = null;

            foreach ($history as $week) {
                if (intval($week['event']) === $gw) {
                    $weekData = $week;
                    break;
                }
            }

            if (!$weekData) continue;

            $points = intval($weekData['points'] ?? 0);
            $transferCost = intval($weekData['event_transfers_cost'] ?? 0);
            $benchPoints = intval($weekData['points_on_bench'] ?? 0);
            $net = $points - $transferCost;

            $teamData = isset($picksData[$managerId]) ? json_decode($picksData[$managerId], true) : [];
            $viceCaptainPoints = getViceCaptainPointsFast($teamData, $liveData);
            $goalsScored = getTeamGoalsFast($teamData, $liveData);

            // Step 8: Insert or update database
            if (isset($existing[$managerId][$gw])) {
                $stmt = $db->query("UPDATE manager_history SET points='$points', transfer_cost='$transferCost', net_points='$net', bench_points='$benchPoints', vice_captain_points='$viceCaptainPoints', team_goals='$goalsScored' WHERE manager_id='$managerId' AND event='$gw' ");
            } else {
                $stmt = $db->query("INSERT INTO manager_history (manager_id, event, points, transfer_cost, net_points, bench_points, vice_captain_points, team_goals) VALUES ('$managerId', '$gw', '$points', '$transferCost', '$net', '$benchPoints', '$viceCaptainPoints', '$goalsScored')");
            }
        }
    }
    updateAllResults();
}





function getAllGwStatuses()
{
    $url = "https://fantasy.premierleague.com/api/bootstrap-static/";

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    curl_close($ch);

    if (!$response) {
        return [];
    }

    $data = json_decode($response, true);
    $statuses = [];

    foreach ($data['events'] as $event) {
        if ($event['finished']) {
            $statuses[$event['id']] = "Closed";
        } elseif ($event['is_current']) {
            $statuses[$event['id']] = "Live";
        } else {
            $statuses[$event['id']] = "Pending";
        }
    }

    return $statuses; // array: [ gwId => "Closed"/"Live"/"Pending" ]
}

function sumManagersPoints(array $managerIds, int $gw, array $histories): int
{
    $total = 0;

    foreach ($managerIds as $managerId) {
        // Skip invalid IDs
        if (!$managerId || !isset($histories[$managerId])) continue;

        $history = $histories[$managerId];
        if (!isset($history['current'])) continue;

        // Use pre-fetched data instead of querying DB or refetching
        foreach ($history['current'] as $week) {
            if (intval($week['event']) === $gw) {
                $points = intval($week['points'] ?? 0);
                $transferCost = intval($week['event_transfers_cost'] ?? 0);
                $net = $points - $transferCost;

                $total += $net;
                break; // stop after finding this gameweek
            }
        }
    }

    return $total;
}



function applyGoalRules($homePoints, $awayPoints)
{
    global $db;
    // load goal rules
    $goalRules = [];
    $qr = $db->query("SELECT id, min_points, max_points, goals FROM goal_rules ORDER BY min_points ASC");
    while ($r = mysqli_fetch_assoc($qr)) {
        $goalRules[] = [
            'id' => intval($r['id']),
            'min' => intval($r['min_points']),
            'max' => intval($r['max_points']),
            'goals' => intval($r['goals'])
        ];
    }

    if (empty($goalRules)) {
        return ['error' => 'No goal rules found'];
    }

    // helper: find band by points
    $findBand = function ($points) use ($goalRules) {
        foreach ($goalRules as $band) {
            if ($points >= $band['min'] && $points <= $band['max']) {
                return $band;
            }
        }
        return null;
    };

    $homeBand = $findBand($homePoints);
    $awayBand = $findBand($awayPoints);

    if (!$homeBand || !$awayBand) {
        return ['error' => 'Points do not fall into any goal band', 'home_band' => $homeBand, 'away_band' => $awayBand];
    }

    // load settings in one query
    $settings = [
        'same_band_margin' => 5,
        'diff_band_margin' => 5,
        'extra_goal_enabled' => 0,
        'extra_goal_margin' => 10
    ];

    $query = $db->query("SELECT diff_band_margin, same_band_margin, extra_goal_enabled, extra_goal_margin FROM settings ORDER BY id ASC");

    $rules = [];
    $settings = [];
    while ($row = mysqli_fetch_assoc($query)) {
        $rules[] = $row;
    }

    $settings = $rules[0];

    $homeGoals = $homeBand['goals'];
    $awayGoals = $awayBand['goals'];
    $margin = abs($homePoints - $awayPoints);
    $explanation = '';

    // same band logic
    if ($homeBand['id'] === $awayBand['id']) {
        if ($margin > intval($settings['same_band_margin'])) {
            if ($homePoints > $awayPoints) {
                $homeGoals += 1;
                $explanation = "Same band, margin > {$settings['same_band_margin']} → +1 to Home";
            } else {
                $awayGoals += 1;
                $explanation = "Same band, margin > {$settings['same_band_margin']} → +1 to Away";
            }
        } else {
            $explanation = "Same band, margin ≤ {$settings['same_band_margin']} → Draw";
        }
    } else {
        // different bands
        if ($margin <= intval($settings['diff_band_margin'])) {
            $drawGoals = min($homeGoals, $awayGoals);
            $homeGoals = $drawGoals;
            $awayGoals = $drawGoals;
            $explanation = "Different bands, margin ≤ {$settings['diff_band_margin']} → Forced draw at lower band goals";
        } else {
            $explanation = "Different bands, margin > {$settings['diff_band_margin']} → Goals by band";
        }
    }

    // extra goal rule per your request:
    // if extra_goal_enabled == 0 then add one goal to higher points side when margin <= extra_goal_margin
    if (intval($settings['extra_goal_enabled']) === 0) {
        $egm = intval($settings['extra_goal_margin']);
        if ($margin <= $egm && $margin > 0) {
            if ($homePoints > $awayPoints) {
                $homeGoals += 1;
                $explanation .= " | Extra rule: +1 to Home (margin ≤ {$egm})";
            } elseif ($awayPoints > $homePoints) {
                $awayGoals += 1;
                $explanation .= " | Extra rule: +1 to Away (margin ≤ {$egm})";
            }
        }
    }

    return [
        'home_points' => $homePoints,
        'away_points' => $awayPoints,
        'home_goals' => intval($homeGoals),
        'away_goals' => intval($awayGoals),
        'home_band' => $homeBand,
        'away_band' => $awayBand,
        'margin' => $margin,
        'explanation' => $explanation
    ];
}

function saveResult($fixtureId, $gameweek, $homeTeamId, $awayTeamId, $homePoints, $awayPoints, $homeGoals, $awayGoals, $status, $explanation, $division)
{
    global $db;
    extract($_POST);
    // Check if result already exists
    $check = $db->query("SELECT id FROM results WHERE fixture_id = '$fixtureId' AND gameweek = '$gameweek' AND division = '$division'");
    $existing = mysqli_num_rows($check);

    if ($existing >= 1) {
        // Update existing
        $stmt = $db->query("UPDATE results SET home_points = '$homePoints', away_points = '$awayPoints', home_goals = '$homeGoals', away_goals = '$awayGoals', status = '$status', explanation = '$explanation', updated_at = NOW() WHERE fixture_id = '$fixtureId' AND gameweek = '$gameweek' AND division = '$division' ");
    } else {
        // Insert new
        $stmt = $db->query("INSERT INTO results 
            (fixture_id, gameweek, home_team, away_team, home_points, away_points, home_goals, away_goals, status, explanation, division) 
            VALUES ('$fixtureId', '$gameweek', '$homeTeamId', '$awayTeamId', '$homePoints', '$awayPoints', '$homeGoals', '$awayGoals', '$status', '$explanation', '$division')");
    }
}


function multiFetchFplHistories2(array $managerIds): array
{
    if (empty($managerIds)) return [];

    $mh = curl_multi_init();
    $handles = [];
    $responses = [];

    // 1. Create all requests
    foreach ($managerIds as $id) {
        $url = "https://fantasy.premierleague.com/api/entry/{$id}/history/";
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_CONNECTTIMEOUT => 5,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_USERAGENT => "Mozilla/5.0"
        ]);
        curl_multi_add_handle($mh, $ch);
        $handles[$id] = $ch;
    }

    // 2. Run all in parallel
    $running = null;
    do {
        curl_multi_exec($mh, $running);
        curl_multi_select($mh);
    } while ($running > 0);

    // 3. Collect results
    foreach ($handles as $id => $ch) {
        $raw = curl_multi_getcontent($ch);
        $decoded = json_decode($raw, true);

        if (is_array($decoded) && isset($decoded['current'])) {
            $responses[$id] = $decoded;
        } else {
            $responses[$id] = null; // store null for failed fetch
        }

        curl_multi_remove_handle($mh, $ch);
        curl_close($ch);
    }

    curl_multi_close($mh);
    return $responses;
}





function updateAllResults()
{
    global $db;

    $latestGW = getLatestGameweek();
    $gwStatuses = getAllGwStatuses();

    // 1. Collect all managers once
    $managers = [];
    $teamRes = $db->query("SELECT manager1, manager2, manager3, manager4, manager5 FROM teams");
    while ($row = mysqli_fetch_assoc($teamRes)) {
        foreach ($row as $m) {
            $m = intval($m);
            if ($m > 0) $managers[$m] = true;
        }
    }
    $managerIds = array_keys($managers);

    // 2. Fetch all histories in parallel instead of sequentially
    $managers = multiFetchFplHistories2($managerIds);

    // 3. Preload all manager_history records once
    $historyCache = [];
    $res = $db->query("SELECT * FROM manager_history");
    while ($r = mysqli_fetch_assoc($res)) {
        $historyCache[$r['manager_id']][$r['event']] = $r;
    }

    // 4. Clear goalscorers once
    $db->query("TRUNCATE TABLE goalscorers");

    // 5. Preload all fixtures once, grouped by GW
    $fixturesByGw = [];
    $fx = $db->query("SELECT * FROM fixtures WHERE gameweek <= $latestGW ORDER BY gameweek");
    while ($f = mysqli_fetch_assoc($fx)) {
        $fixturesByGw[$f['gameweek']][] = $f;
    }

    // 6. Main loop
    foreach ($fixturesByGw as $gw => $fixtures) {
        $status = $gwStatuses[$gw] ?? "Pending";

        foreach ($fixtures as $f) {
            $division = getTeamDivision($f['home_team_id']);
            $home_managers = getTeamManagers($f['home_team_id']);
            $away_managers = getTeamManagers($f['away_team_id']);

            if (is_string($home_managers)) {
                $try = json_decode($home_managers, true);
                $home_managers = is_array($try) ? $try : [];
            }

            if (is_string($away_managers)) {
                $try = json_decode($away_managers, true);
                $away_managers = is_array($try) ? $try : [];
            }

            $homePoints = sumManagersPoints($home_managers, $gw, $managers);
            $awayPoints = sumManagersPoints($away_managers, $gw, $managers);

            $goals = applyGoalRules($homePoints, $awayPoints);

            saveResult(
                $f['id'],
                $gw,
                $f['home_team_id'],
                $f['away_team_id'],
                $homePoints,
                $awayPoints,
                $goals['home_goals'],
                $goals['away_goals'],
                $status,
                $goals['explanation'],
                $division
            );

            // handle goalscorer logic
            if ($goals['home_goals'] == $goals['away_goals']) continue;

            $winning_team_id = $goals['home_goals'] > $goals['away_goals'] ? $f['home_team_id'] : $f['away_team_id'];
            $winning_managers = getTeamManagers($winning_team_id);

            if (is_string($winning_managers)) {
                $try = json_decode($winning_managers, true);
                $winning_managers = is_array($try) ? $try : [];
            }

            $top_manager = null;
            $top_points = -INF;
            $top_vc = -INF;
            $top_bench = -INF;
            $top_goals = -INF;

            // Use cached manager_history data
            foreach ($winning_managers as $m) {
                $h = $historyCache[$m][$gw] ?? null;
                if (!$h) continue;

                $net = $h['net_points'];
                $vc = $h['vice_captain_points'];
                $bench = $h['bench_points'];
                $tgoals = $h['team_goals'];

                if (
                    $net > $top_points ||
                    ($net == $top_points && (
                        $vc > $top_vc ||
                        ($vc == $top_vc && (
                            $bench > $top_bench ||
                            ($bench == $top_bench && $tgoals > $top_goals)
                        ))
                    ))
                ) {
                    $top_points = $net;
                    $top_vc = $vc;
                    $top_bench = $bench;
                    $top_goals = $tgoals;
                    $top_manager = $m;
                }
            }

            if ($top_manager) {
                $fixture_id = $f['id'];
                $stmt = $db->query("INSERT INTO goalscorers (gameweek, fixture_id, team_id, manager_id, points, division) VALUES ('$gw', '$fixture_id', '$winning_team_id', '$top_manager', '$top_points', '$division')");
            }
        }
    }
}


updateManagerHistories();
// updateManagerHistory(473236, 4);
// updateAllResults();
