<?php

include('../connect.php');
// championscup_overview.php
function displayChampionsCupOverview($season = '2025') {
    global $db;
    
    echo "<div class='cup-overview'>";
    echo "<h1>🏆 Maestros Champions Cup $season</h1>";
    
    // Current stage indicator
    displayCurrentStage($season);
    
    // Next fixtures
    displayNextFixtures($season);
    
    // Latest results
    displayLatestResults($season);
    
    // Group standings summary
    displayGroupsSummary($season);
    
    // Top scorers preview
    displayTopScorersPreview($season);
    
    // Key statistics
    displayCupStatistics($season);
    
    echo "</div>";
}
displayChampionsCupOverview();


// championscup_display_functions.php

/**
 * MAIN OVERVIEW FUNCTIONS
 */
function displayCurrentStage($season = '2025') {
    global $db;
    
    // Determine current stage based on gameweek
    $current_gw = getLatestGameweek();
    $stage = getCupStageByGameweek($current_gw);
    
    echo "<div class='current-stage'>";
    echo "<h2>Current Stage: $stage</h2>";
    
    if ($stage == 'Group Stage') {
        $matchday = getCurrentGroupMatchday($current_gw);
        echo "<p>Matchday $matchday of 6</p>";
    } elseif ($stage == 'Knockout Stage') {
        $round = getCurrentKnockoutRound($current_gw);
        echo "<p>$round in progress</p>";
    } elseif ($stage == 'Completed') {
        $champion = getCurrentChampion($season);
        echo "<p>🏆 Champion: $champion</p>";
    }
    
    echo "</div>";
}

function displayNextFixtures($season = '2025') {
    global $db;
    
    $current_gw = getLatestGameweek();
    $next_gw = getNextCupGameweek($current_gw);
    
    $fixtures = $db->query("SELECT f.*, t1.team_name as home_team, t2.team_name as away_team, 
                                   cg.group_name, f.stage 
                            FROM cup_fixtures f 
                            JOIN teams t1 ON f.home_team_id = t1.id 
                            JOIN teams t2 ON f.away_team_id = t2.id 
                            LEFT JOIN cup_group_teams cgt ON f.home_team_id = cgt.team_id 
                            LEFT JOIN cup_groups cg ON cgt.group_id = cg.id 
                            WHERE f.gameweek = $next_gw 
                            AND f.season_year = '$season' 
                            AND f.status = 'Pending' 
                            ORDER BY f.stage, cg.group_name 
                            LIMIT 8");
    
    echo "<div class='next-fixtures'>";
    echo "<h3>📅 Next Fixtures (GW$next_gw)</h3>";
    
    if ($fixtures->num_rows == 0) {
        echo "<p>No upcoming fixtures</p>";
        return;
    }
    
    echo "<div class='fixtures-list'>";
    while ($fixture = $fixtures->fetch_assoc()) {
        $stage_label = $fixture['stage'] == 'Group' ? "Group {$fixture['group_name']}" : $fixture['stage'];
        echo "<div class='fixture-item'>";
        echo "<span class='stage'>$stage_label</span>";
        echo "<span class='teams'>{$fixture['home_team']} vs {$fixture['away_team']}</span>";
        echo "</div>";
    }
    echo "</div>";
    echo "</div>";
}

function displayLatestResults($season = '2025') {
    global $db;
    
    $latest_gw = $db->query("SELECT MAX(gameweek) as latest_gw FROM cup_fixtures 
                            WHERE status = 'Closed' AND season_year = '$season'")->fetch_assoc()['latest_gw'];
    
    if (!$latest_gw) return;
    
    $results = $db->query("SELECT f.*, t1.team_name as home_team, t2.team_name as away_team, 
                                  cg.group_name, f.stage 
                           FROM cup_fixtures f 
                           JOIN teams t1 ON f.home_team_id = t1.id 
                           JOIN teams t2 ON f.away_team_id = t2.id 
                           LEFT JOIN cup_group_teams cgt ON f.home_team_id = cgt.team_id 
                           LEFT JOIN cup_groups cg ON cgt.group_id = cg.id 
                           WHERE f.gameweek = $latest_gw 
                           AND f.season_year = '$season' 
                           AND f.status = 'Closed' 
                           ORDER BY f.stage, cg.group_name 
                           LIMIT 6");
    
    echo "<div class='latest-results'>";
    echo "<h3>⚽ Latest Results (GW$latest_gw)</h3>";
    
    if ($results->num_rows == 0) {
        echo "<p>No recent results</p>";
        return;
    }
    
    echo "<div class='results-list'>";
    while ($result = $results->fetch_assoc()) {
        $stage_label = $result['stage'] == 'Group' ? "Group {$result['group_name']}" : $result['stage'];
        echo "<div class='result-item'>";
        echo "<span class='stage'>$stage_label</span>";
        echo "<span class='score'>{$result['home_team']} {$result['home_goals']}-{$result['away_goals']} {$result['away_team']}</span>";
        echo "</div>";
    }
    echo "</div>";
    echo "</div>";
}

function displayGroupsSummary($season = '2025') {
    global $db;
    
    $groups = $db->query("SELECT group_name FROM cup_groups WHERE season_year = '$season' ORDER BY group_name");
    
    echo "<div class='groups-summary'>";
    echo "<h3>🏆 Group Standings</h3>";
    echo "<div class='groups-grid'>";
    
    while ($group = $groups->fetch_assoc()) {
        $group_name = $group['group_name'];
        $standings = computeGroupStandingsByGroupName($group_name, $season);
        
        echo "<div class='group-summary'>";
        echo "<h4>Group $group_name</h4>";
        
        // Show top 2 teams
        for ($i = 0; $i < min(2, count($standings)); $i++) {
            $team = $standings[$i];
            $qualifying = $i < 2 ? 'qualifying' : '';
            echo "<div class='team-standing $qualifying'>";
            echo "<span class='position'>{$team['position']}.</span>";
            echo "<span class='team-name'>{$team['team_name']}</span>";
            echo "<span class='points'>{$team['points']}pts</span>";
            echo "</div>";
        }
        
        echo "</div>";
    }
    
    echo "</div>";
    echo "</div>";
}

function displayTopScorersPreview($season = '2025') {
    global $db;
    
    $top_scorers = $db->query("SELECT g.manager_id, g.team_id, SUM(g.points) as total_goals, 
                                      t.name as team_name, m.manager_name 
                               FROM cup_goalscorers g 
                               JOIN teams t ON g.team_id = t.id 
                               JOIN managers m ON g.manager_id = m.id 
                               WHERE g.season_year = '$season' 
                               GROUP BY g.manager_id, g.team_id 
                               ORDER BY total_goals DESC 
                               LIMIT 5");
    
    echo "<div class='top-scorers-preview'>";
    echo "<h3>🎯 Top Scorers</h3>";
    
    if ($top_scorers->num_rows == 0) {
        echo "<p>No goalscorers yet</p>";
        return;
    }
    
    echo "<div class='scorers-list'>";
    $position = 1;
    while ($scorer = $top_scorers->fetch_assoc()) {
        echo "<div class='scorer-item'>";
        echo "<span class='position'>$position.</span>";
        echo "<span class='scorer-name'>{$scorer['manager_name']}</span>";
        echo "<span class='team'>{$scorer['team_name']}</span>";
        echo "<span class='goals'>{$scorer['total_goals']} ⚽</span>";
        echo "</div>";
        $position++;
    }
    echo "</div>";
    echo "</div>";
}

function displayCupStatistics($season = '2025') {
    global $db;
    
    // Basic stats
    $total_matches = $db->query("SELECT COUNT(*) as total FROM cup_fixtures WHERE season_year = '$season' AND status = 'Closed'")->fetch_assoc()['total'];
    $total_goals = $db->query("SELECT SUM(home_goals + away_goals) as total FROM cup_fixtures WHERE season_year = '$season' AND status = 'Closed'")->fetch_assoc()['total'];
    $avg_goals = $total_matches > 0 ? round($total_goals / $total_matches, 2) : 0;
    
    echo "<div class='cup-statistics'>";
    echo "<h3>📈 Tournament Statistics</h3>";
    echo "<div class='stats-grid'>";
    echo "<div class='stat-item'><span class='stat-value'>$total_matches</span><span class='stat-label'>Matches</span></div>";
    echo "<div class='stat-item'><span class='stat-value'>$total_goals</span><span class='stat-label'>Goals</span></div>";
    echo "<div class='stat-item'><span class='stat-value'>$avg_goals</span><span class='stat-label'>Avg Goals/Match</span></div>";
    echo "</div>";
    echo "</div>";
}

/**
 * HELPER FUNCTIONS FOR OVERVIEW
 */
function getCupStageByGameweek($gameweek) {
    if ($gameweek < 12) return 'Not Started';
    if ($gameweek <= 22) return 'Group Stage';
    if ($gameweek <= 38) return 'Knockout Stage';
    return 'Completed';
}

function getCurrentGroupMatchday($gameweek) {
    $matchdays = [
        12 => 1, 14 => 2, 16 => 3, 18 => 4, 20 => 5, 22 => 6
    ];
    return $matchdays[$gameweek] ?? 0;
}

function getCurrentKnockoutRound($gameweek) {
    $rounds = [
        24 => 'Round of 16', 26 => 'Round of 16',
        28 => 'Quarter Finals', 30 => 'Quarter Finals', 
        32 => 'Semi Finals', 34 => 'Semi Finals',
        38 => 'Final'
    ];
    return $rounds[$gameweek] ?? 'Knockout Stage';
}

function getNextCupGameweek($current_gw) {
    $cup_gameweeks = [12, 14, 16, 18, 20, 22, 24, 26, 28, 30, 32, 34, 36, 38];
    
    foreach ($cup_gameweeks as $gw) {
        if ($gw > $current_gw) {
            return $gw;
        }
    }
    return end($cup_gameweeks);
}

function computeCupGroupStandings($group_id, $season = '2025')
{
    global $db;

    // Get teams in group
    $teams_res = $db->query("SELECT cgt.team_id, t.team_name 
                            FROM cup_group_teams cgt 
                            JOIN teams t ON cgt.team_id = t.id 
                            WHERE cgt.group_id = $group_id");

    $standings = [];
    while ($team = $teams_res->fetch_assoc()) {
        $standings[$team['team_id']] = [
            'team_id' => $team['team_id'],
            'team_name' => $team['team_name'],
            'played' => 0,
            'won' => 0,
            'drawn' => 0,
            'lost' => 0,
            'gf' => 0,
            'ga' => 0,
            'gd' => 0,
            'points' => 0
        ];
    }

    // Get all fixtures for this group
    $teamIds = array_keys($standings);
    $teamIdsStr = implode(',', $teamIds);

    $fixtures = $db->query("SELECT * FROM cup_fixtures 
                           WHERE stage = 'Group' 
                           AND season_year = '" . $db->real_escape_string($season) . "' 
                           AND home_team_id IN ($teamIdsStr) 
                           AND away_team_id IN ($teamIdsStr) 
                           AND status = 'Closed'");

    while ($fx = $fixtures->fetch_assoc()) {
        $home = $fx['home_team_id'];
        $away = $fx['away_team_id'];
        $hg = intval($fx['home_goals']);
        $ag = intval($fx['away_goals']);

        $standings[$home]['played']++;
        $standings[$away]['played']++;
        $standings[$home]['gf'] += $hg;
        $standings[$home]['ga'] += $ag;
        $standings[$away]['gf'] += $ag;
        $standings[$away]['ga'] += $ag;

        if ($hg > $ag) {
            $standings[$home]['won']++;
            $standings[$home]['points'] += 3;
            $standings[$away]['lost']++;
        } elseif ($ag > $hg) {
            $standings[$away]['won']++;
            $standings[$away]['points'] += 3;
            $standings[$home]['lost']++;
        } else {
            $standings[$home]['drawn']++;
            $standings[$away]['drawn']++;
            $standings[$home]['points'] += 1;
            $standings[$away]['points'] += 1;
        }
    }

    // Calculate GD and sort (using your existing sorting logic)
    foreach ($standings as &$team) {
        $team['gd'] = $team['gf'] - $team['ga'];
    }

    uasort($standings, function ($a, $b) {
        if ($a['points'] !== $b['points']) return $b['points'] - $a['points'];
        if ($a['gd'] !== $b['gd']) return $b['gd'] - $a['gd'];
        if ($a['gf'] !== $b['gf']) return $b['gf'] - $a['gf'];
        return 0;
    });

    // Add positions
    $position = 1;
    foreach ($standings as &$team) {
        $team['position'] = $position;
        $position++;
    }

    return $standings;
}

function computeGroupStandingsByGroupName($group_name, $season) {
    global $db;
    
    $group = $db->query("SELECT id FROM cup_groups WHERE group_name = '$group_name' AND season_year = '$season'")->fetch_assoc();
    if (!$group) return [];
    
    return computeCupGroupStandings($group['id'], $season);
}

function getCurrentChampion($season) {
    global $db;
    
    $champion = $db->query("SELECT t.name 
                           FROM cup_fixtures f 
                           JOIN teams t ON f.winner_team_id = t.id 
                           WHERE f.stage = 'F' 
                           AND f.season_year = '$season' 
                           LIMIT 1")->fetch_assoc();
    
    return $champion ? $champion['name'] : 'TBD';
}
?>